import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:get/get.dart';
import 'package:viser_remit/core/helper/string_format_helper.dart';
import 'package:viser_remit/core/route/route.dart';
import 'package:viser_remit/core/utils/style.dart';
import 'package:viser_remit/data/controller/send_money/send_money_controller.dart';
import 'package:viser_remit/data/repo/send_money/send_money_repo.dart';
import 'package:viser_remit/view/components/status/history_status_section.dart';

import '../../../core/helper/date_converter.dart';
import '../../../core/utils/dimensions.dart';
import '../../../core/utils/my_color.dart';
import '../../../core/utils/my_strings.dart';
import '../../../core/utils/util.dart';
import '../../../data/controller/send_money_history/send_money_history_controller.dart';
import '../../../data/services/api_client.dart';
import '../../components/app-bar/viser_remit_app_bar.dart';
import '../../components/buttons/rounded_button.dart';
import '../../components/divider/custom_divider.dart';
import '../../components/snack_bar/show_custom_snackbar.dart';
import '../../components/status/send_money_status_chips.dart';

class SendMoneyHistoryDetailsScreen extends StatefulWidget {
  const SendMoneyHistoryDetailsScreen({super.key});

  @override
  State<SendMoneyHistoryDetailsScreen> createState() => _SendMoneyHistoryDetailsScreenState();
}

class _SendMoneyHistoryDetailsScreenState extends State<SendMoneyHistoryDetailsScreen> {
  @override
  void initState() {
    Get.put(ApiClient(sharedPreferences: Get.find()));
    Get.put(SendMoneyRepo(apiClient: Get.find()));
    Get.find<SendMoneyHistoryController>();
    Get.put(SendMoneyController(sendMoneyRepo: Get.find()));
    super.initState();
  }

  @override
  Widget build(BuildContext context) {
    return GetBuilder<SendMoneyHistoryController>(builder: (controller) {
      var historyDetails = controller.sendMoneyHistoryList[controller.currentIndex];

      return Scaffold(
        appBar:  ViserRemitAppBar(
          title: MyStrings.details.tr,
          isElevation: true,
        ),
        body: SingleChildScrollView(
          physics: const BouncingScrollPhysics(),
          padding: const EdgeInsets.symmetric(horizontal: Dimensions.space10, vertical: Dimensions.space10),
          child: Column(
            children: [
              //Main Details

              Container(
                padding: const EdgeInsetsDirectional.symmetric(horizontal: Dimensions.space15, vertical: Dimensions.space15),
                decoration: BoxDecoration(
                  borderRadius: BorderRadius.circular(Dimensions.space10),
                  color: MyColor.getCardBgColor(),
                  boxShadow: MyUtils.getCardShadow(),
                ),
                child: Column(
                  children: [
                    Row(
                      mainAxisSize: MainAxisSize.min,
                      children: [
                        Expanded(child: Text(MyStrings.recipientsName.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor))),
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                              child: Text(
                                historyDetails.recipient?.name.toString() ?? "",
                                style: mediumLarge,
                              ),
                            ),
                          ],
                        ))
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Expanded(child: Text(MyStrings.recipientMobileNo.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor))),
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                              child: Text(
                                "+${historyDetails.recipient?.mobile}",
                                style: mediumLarge,
                              ),
                            ),
                          ],
                        ))
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Expanded(child: Text(MyStrings.recipientAddressS.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor))),
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                                child: Text(
                              historyDetails.recipient?.address ?? "".tr,
                              style: mediumLarge,
                            )),
                          ],
                        ))
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Text(MyStrings.recipientCountry.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor)),
                        const Spacer(),
                        Text(
                          historyDetails.recipientCountry?.name.toString() ?? "".tr,
                          style: mediumLarge,
                        )
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Text(MyStrings.mtcn.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor)),
                        const Spacer(),
                        Text(
                          "#${historyDetails.mtcnNumber}",
                          style: mediumLarge,
                        )
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Expanded(child: Text(MyStrings.transactionNumber.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor))),
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                              child: GestureDetector(
                                onTap: () async {
                                  await Clipboard.setData(ClipboardData(text: "${historyDetails.trx}"));
                                  CustomSnackBar.success(successList: [MyStrings.copied.tr]);
                                },
                                child: Text(
                                  "#${historyDetails.trx}",
                                  style: mediumLarge,
                                ),
                              ),
                            ),
                          ],
                        ))
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Text(MyStrings.deliveryMethod.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor)),
                        const Spacer(),
                        Text(
                          historyDetails.countryDeliveryMethod?.deliveryMethod?.name.toString() ?? MyStrings.agent,
                          style: mediumLarge,
                        ),
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Expanded(child: Text(MyStrings.status.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor))), //todo
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                                child: SendMoneyHistoryStatusSection(
                              status: historyDetails.status.toString(),
                              isDepositAvailable: false,
                            )),
                          ],
                        ))
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Expanded(child: Text(MyStrings.payment.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor))), //todo
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                                child: HistoryStatusSection(
                              status: historyDetails.paymentStatus.toString(),
                              isDepositAvailable: historyDetails.deposit == null ? false : true,
                            )),
                          ],
                        ))
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Expanded(child: Text(MyStrings.paymentVia.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor))), //todo
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                                child: Text(
                              historyDetails.deposit?.gateway?.name.toString().tr ?? MyStrings.refundWallet.tr,
                              style: mediumLarge,
                            )),
                          ],
                        ))
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Expanded(child: Text(MyStrings.sentAt.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor))),
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                              child: Text(
                                "${DateConverter.formatDepositTimeWithAmFormat(historyDetails.createdAt.toString())}  ${DateConverter.isoStringToLocalTimeOnly(historyDetails.createdAt.toString())}",
                                style: mediumLarge,
                              ),
                            ),
                          ],
                        ))
                      ],
                    ),
                  ],
                ),
              ),

              const SizedBox(
                height: Dimensions.space20,
              ),
              //Others
              Container(
                padding: const EdgeInsetsDirectional.symmetric(horizontal: Dimensions.space15, vertical: Dimensions.space15),
                decoration: BoxDecoration(
                  borderRadius: BorderRadius.circular(Dimensions.space10),
                  color: MyColor.getCardBgColor(),
                  boxShadow: MyUtils.getCardShadow(),
                ),
                child: Column(
                  children: [
                    Row(
                      mainAxisAlignment: MainAxisAlignment.spaceBetween,
                      children: [
                        Text(MyStrings.recipientWillGet.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor)),
                        Text(
                          "${Converter.formatNumber(historyDetails.recipientAmount.toString())} ${historyDetails.recipientCurrency}",
                          style: mediumLarge,
                        )
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Expanded(child: Text(MyStrings.conversionRate.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor))),
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                                child: Text(
                              controller.getConversionRate(),
                              style: mediumLarge,
                            )),
                          ],
                        ))
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Text(MyStrings.sentAmount.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor)),
                        const Spacer(),
                        Text(
                          "${Converter.formatNumber(historyDetails.sendingAmount.toString())} ${historyDetails.sendingCurrency}",
                          style: mediumLarge,
                        )
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Expanded(child: Text(MyStrings.deliveryMethodCharge.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor))),
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                                child: Text(
                              "${Converter.formatNumber(historyDetails.sendingCharge.toString())} ${historyDetails.sendingCurrency}",
                              style: mediumLarge,
                            )),
                          ],
                        ))
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Expanded(child: Text("${MyStrings.sentAmount.tr} ${MyStrings.includingCharge}", style: regularLarge.copyWith(color: MyColor.bodyTextColor))),
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                              child: Text(
                                controller.getSentAmountIncludingCharge(),
                                style: mediumLarge,
                              ),
                            ),
                          ],
                        ))
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Expanded(child: Text(MyStrings.conversionRate.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor))),
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                              child: Text(
                              controller.getConversionRateInBaseCurrency(),
                                style: mediumLarge,
                              ),
                            ),
                          ],
                        ))
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Expanded(child: Text("${MyStrings.sendingAmount.tr} ${MyStrings.includingCharge} ${controller.sendMoneyHistoryRepo.apiClient.getCurrencyOrUsername(isCurrency: true)}", style: regularLarge.copyWith(color: MyColor.bodyTextColor))),
                        Expanded(
                            child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            const SizedBox(),
                            Flexible(
                                child: Text(
                              controller.getSendingAmountIncludingCharge(),
                              style: mediumLarge,
                            )),
                          ],
                        ))
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Text(MyStrings.gatewayCharge.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor)),
                        const Spacer(),
                        Text(
                          "${Converter.formatNumber(historyDetails.deposit?.charge.toString() ?? "0.00")}  ${controller.sendMoneyHistoryRepo.apiClient.getCurrencyOrUsername(isCurrency: true)}",
                          style: mediumLarge,
                        ) //todo
                      ],
                    ),
                    const CustomDivider(),
                    Row(
                      children: [
                        Text(MyStrings.totalPayableAmount.tr, style: regularLarge.copyWith(color: MyColor.bodyTextColor)),
                        const Spacer(),
                        Text(
                          controller.getSendingAmountIncludingChargeAndGetwayChanrage(),
                          style: mediumLarge,
                        ) //todo
                      ],
                    ),
                  ],
                ),
              ),
              const SizedBox(height: Dimensions.space20),
              historyDetails.paymentStatus != '1'
                  ? RoundedButton(
                      fontSize: 16,
                      verticalPadding: 15,
                      text: MyStrings.payNow.tr,
                      press: () {
                        Get.find<SendMoneyController>().setTrxNumber(historyDetails.trx.toString());
                        Get.toNamed(RouteHelper.newDepositScreenScreen, arguments: controller.getAmountIncludingCharge());
                      },
                    )
                  : const SizedBox.shrink(),
                  const SizedBox(height: Dimensions.space40),
            ],
          ),
        ),
      );
    });
  }
}
